///////////////////////////////////////////////////////////////////////////////
// Name:        wx/aui/toolbar.h
// Purpose:     Documentation of wxAuiToolBar and related classes.
// Created:     2011-01-17
// Copyright:   (c) 2011 wxWidgets development team
// Licence:     wxWindows licence
///////////////////////////////////////////////////////////////////////////////

/**
    wxAuiToolBarStyle is part of the wxAUI class framework, used to define the appearance of a wxAuiToolBar.

    See also @ref overview_aui.

    @library{wxaui}
    @category{aui}
  */
enum wxAuiToolBarStyle
{
    /**
      Shows the text in the toolbar buttons; by default only icons are shown.
    */
    wxAUI_TB_TEXT          = 1 << 0,

    /**
      Don't show tooltips on wxAuiToolBar items.
    */
    wxAUI_TB_NO_TOOLTIPS   = 1 << 1,

    /**
      Do not auto-resize the wxAuiToolBar.
    */
    wxAUI_TB_NO_AUTORESIZE = 1 << 2,

    /**
      Shows a gripper on the wxAuiToolBar.
    */
    wxAUI_TB_GRIPPER       = 1 << 3,

    /**
      The wxAuiToolBar can contain overflow items.
    */
    wxAUI_TB_OVERFLOW      = 1 << 4,

    /**
      Using this style forces the toolbar to be vertical and be only dockable to the left or right sides of the window whereas by default it can be horizontal or vertical and be docked anywhere.
    */
    wxAUI_TB_VERTICAL      = 1 << 5,

    /**
      Shows the text and the icons alongside, not vertically stacked.

      This style must be used with wxAUI_TB_TEXT and can't be used with
      wxAUI_TB_VERTICAL.
    */
    wxAUI_TB_HORZ_LAYOUT   = 1 << 6,

    /**
      Analogous to wxAUI_TB_VERTICAL, but forces the toolbar to be horizontal, docking to the top or bottom of the window.
    */
    wxAUI_TB_HORIZONTAL    = 1 << 7,


    /**
      Draw a plain background (based on parent) instead of the default gradient background.

      @since 2.9.5
    */
    wxAUI_TB_PLAIN_BACKGROUND = 1 << 8,

    /**
        For vertical toolbars, put icon above vertical text.

        This also rotates the icon by 90 degrees clockwise for consistency with
        the text rotation angle when using wxAuiTextDirection::BottomToTop.

        @since 3.3.2
    */
    wxAUI_TB_VERT_LAYOUT_DOWN = 1 << 9,

    /**
        For vertical toolbars, put icon below vertical text.

        This also rotates the icon by 90 degrees counter clockwise for
        consistency with the text rotation angle when using
        wxAuiTextDirection::TopToBottom.

        @since 3.3.2
    */
    wxAUI_TB_VERT_LAYOUT_UP = 1 << 10,

    /**
       Shows the text alongside the icons, not vertically stacked.
    */
    wxAUI_TB_HORZ_TEXT     = (wxAUI_TB_HORZ_LAYOUT | wxAUI_TB_TEXT),

    /**
       Shows the text for vertical toolbars, with icon above text.

       @since 3.3.2
    */
    wxAUI_TB_VERT_TEXT_DOWN = (wxAUI_TB_VERT_LAYOUT_DOWN | wxAUI_TB_TEXT),

    /**
       Shows the text for vertical toolbars, with icon below text.

       @since 3.3.2
    */
    wxAUI_TB_VERT_TEXT_UP = (wxAUI_TB_VERT_LAYOUT_UP | wxAUI_TB_TEXT),

    /**
        Rotate the icon to match the text orientation.

        When this flag is set, the icons in the toolbar items are rotated by 90
        degrees clockwise or counter clockwise to match the text orientation,
        i.e. the logical top of the icon is aligned with the top of the text
        (and so is on the left or right side of the toolbar depending on
        whether it is positioned along the left or right side of the window).

        This flag only has effect for vertical toolbars when used together with
        wxAUI_TB_VERT_LAYOUT_DOWN or wxAUI_TB_VERT_LAYOUT_UP.

        @since 3.3.2
     */
    wxAUI_TB_ROTATE_ICON_WITH_TEXT = 1 << 11,

    /**
        Mask including both orientation flags.

        This value can't be used as a style.
     */
    wxAUI_ORIENTATION_MASK = (wxAUI_TB_VERTICAL | wxAUI_TB_HORIZONTAL),

    /**
      By default only icons are shown.
    */
    wxAUI_TB_DEFAULT_STYLE = 0

};

/**
    wxAuiToolBarArtSetting

    @library{wxaui}
    @category{aui}
*/
enum wxAuiToolBarArtSetting
{
    /**
      wxAuiToolBar separator size.
    */
    wxAUI_TBART_SEPARATOR_SIZE = 0,

    /**
      wxAuiToolBar gripper size.
    */
    wxAUI_TBART_GRIPPER_SIZE = 1,

    /**
      Overflow button size in wxAuiToolBar.
    */
    wxAUI_TBART_OVERFLOW_SIZE = 2,

    /**
      Drop down button size in wxAuiToolBar.

      @since 3.1.2
    */
    wxAUI_TBART_DROPDOWN_SIZE = 3

};

/**
  wxAuiToolBarToolTextOrientation

  @library{wxaui}
  @category{aui}
  */
enum wxAuiToolBarToolTextOrientation
{
    /**
      Text in wxAuiToolBar items is left aligned, currently unused/unimplemented.
    */
    wxAUI_TBTOOL_TEXT_LEFT = 0,

    /**
      Text in wxAuiToolBar items is right aligned.
    */
    wxAUI_TBTOOL_TEXT_RIGHT = 1,

    /**
      Text in wxAuiToolBar items is top aligned, currently unused/unimplemented.
    */
    wxAUI_TBTOOL_TEXT_TOP = 2,

    /**
      Text in wxAuiToolBar items is bottom aligned.
    */
    wxAUI_TBTOOL_TEXT_BOTTOM = 3

};

/**
    Direction of the text in wxAuiToolBar items.

    Horizontal toolbars always use LeftToRight direction, whereas vertical
    toolbars may use either TopToBottom or BottomToTop.

    @library{wxaui}
    @category{aui}

    @since 3.3.2
  */
enum class wxAuiTextDirection
{
    /// Default direction always used for horizontal toolbars.
    LeftToRight,

    /// May be used for vertical toolbars, typically on the right side.
    TopToBottom,

    /// May be used for vertical toolbars, typically on the left side.
    BottomToTop
};


/**
    @class wxAuiToolBarEvent

    wxAuiToolBarEvent is used for the events generated by @ref wxAuiToolBar.

    @library{wxaui}
    @category{aui,events}
*/
class wxAuiToolBarEvent : public wxNotifyEvent
{
public:
    wxAuiToolBarEvent(wxEventType commandType = wxEVT_NULL,
                      int winId = 0);
    wxAuiToolBarEvent(const wxAuiToolBarEvent& c);

    /**
      Returns whether the drop down menu has been clicked.
    */
    bool IsDropDownClicked() const;

    /**
      Returns the point where the user clicked with the mouse.
    */
    wxPoint GetClickPoint() const;

    /**
      Returns the wxAuiToolBarItem rectangle bounding the mouse click point.
    */
    wxRect GetItemRect() const;

    /**
      Returns the wxAuiToolBarItem identifier.
    */
    int GetToolId() const;


    void SetDropDownClicked(bool c);
    void SetClickPoint(const wxPoint& p);
    void SetItemRect(const wxRect& r);
    void SetToolId(int toolId);
};

wxEventType wxEVT_AUITOOLBAR_TOOL_DROPDOWN;
wxEventType wxEVT_AUITOOLBAR_OVERFLOW_CLICK;
wxEventType wxEVT_AUITOOLBAR_RIGHT_CLICK;
wxEventType wxEVT_AUITOOLBAR_MIDDLE_CLICK;
wxEventType wxEVT_AUITOOLBAR_BEGIN_DRAG;


/**
    @class wxAuiToolBarItem

    wxAuiToolBarItem is part of the wxAUI class framework, representing a toolbar element.

    See also @ref wxAuiToolBar and @ref overview_aui.

    It has a unique id (except for the separators which always have id = -1), the
    style (telling whether it is a normal button, separator or a control), the
    state (toggled or not, enabled or not) and short and long help strings. The
    default implementations use the short help string for the tooltip text which
    is popped up when the mouse pointer enters the tool and the long help string
    for the applications status bar (currently not implemented).

    @library{wxaui}
    @category{aui}
*/
class wxAuiToolBarItem
{
public:

    /**
       Default Constructor
    */
    wxAuiToolBarItem();

    /**
      Assigns the properties of the wxAuiToolBarItem "c" to this.
    */
    wxAuiToolBarItem(const wxAuiToolBarItem& c);

    /**
       Assigns the properties of the wxAuiToolBarItem "c" to this, returning a pointer to this.
    */
    wxAuiToolBarItem& operator=(const wxAuiToolBarItem& c);

    /**
      Assigns the properties of the wxAuiToolBarItem "c" to this.
    */
    void Assign(const wxAuiToolBarItem& c);

    /**
      Assigns a window to the toolbar item.
    */
    void SetWindow(wxWindow* w);
    /**
      Returns the wxWindow* associated to the toolbar item.
    */
    wxWindow* GetWindow();

    /**
      Sets the toolbar item identifier.
    */
    void SetId(int new_id);
    /**
      Returns the toolbar item identifier.
    */
    int GetId() const;

    /**
      Sets the wxAuiToolBarItem kind.
    */
    void SetKind(int new_kind);

    /**
      Returns the toolbar item kind.

      @return one of @c wxITEM_NORMAL, @c wxITEM_CHECK or @c wxITEM_RADIO,
      @c wxITEM_SEPARATOR, @c wxITEM_CONTROL, @c wxITEM_SPACER, @c wxITEM_LABEL,
    */
    int GetKind() const;

    /**
      Set the current state of the toolbar item.

      @param new_state is an or'd combination of flags from wxAuiPaneButtonState
    */
    void SetState(int new_state);
    /**
      Gets the current state of the toolbar item.

      @return an or'd combination of flags from wxAuiPaneButtonState representing the current state
    */
    int GetState() const;

    /**

    */
    void SetSizerItem(wxSizerItem* s);
    /**

    */
    wxSizerItem* GetSizerItem() const;

    /**

    */
    void SetLabel(const wxString& s);
    /**

    */
    const wxString& GetLabel() const;

    /**

    */
    void SetBitmap(const wxBitmapBundle& bmp);
    /**

    */
    wxBitmap GetBitmap() const;

    /**

    */
    void SetDisabledBitmap(const wxBitmapBundle& bmp);
    /**

    */
    wxBitmap GetDisabledBitmap() const;

    /**

    */
    void SetHoverBitmap(const wxBitmapBundle& bmp);
    /**

    */
    wxBitmap GetHoverBitmap() const;

    /**

    */
    void SetShortHelp(const wxString& s);
    /**

    */
    const wxString& GetShortHelp() const;

    /**

    */
    void SetLongHelp(const wxString& s);
    /**

    */
    const wxString& GetLongHelp() const;

    /**

    */
    void SetMinSize(const wxSize& s);
    /**

    */
    const wxSize& GetMinSize() const;

    /**

    */
    void SetSpacerPixels(int s);
    /**

    */
    int GetSpacerPixels() const;

    /**

    */
    void SetProportion(int p);
    /**

    */
    int GetProportion() const;

    /**

    */
    void SetActive(bool b);
    /**

    */
    bool IsActive() const;

    /**
        Set whether this tool has a drop down button.

        This is only valid for wxITEM_NORMAL tools.
    */
    void SetHasDropDown(bool b);
    /**
        Returns whether the toolbar item has an associated drop down button.
    */
    bool HasDropDown() const;

    /**

    */
    void SetSticky(bool b);
    /**

    */
    bool IsSticky() const;

    /**
        Associates a number with the item.

        @param userData Number to associate

        @see GetUserData()
    */
    void SetUserData(long userData);

    /**
        Get number associated with the item.

        @return Associated number

        @see SetUserData()
    */
    long GetUserData() const;

    /**
        Associates a wxObject with the item.

        @param clientData Pointer to the wxObject

        @see GetClientData()

        @since 3.3.0
    */
    void SetClientData(wxObject* clientData);

    /**
        Get wxObject associated with the item.

        @return Pointer to the associated wxObject

        @see SetClientData()

        @since 3.3.0
    */
    wxObject* GetClientData() const;

    /**

    */
    void SetAlignment(int l);
    /**

    */
    int GetAlignment() const;

    /**
        Returns whether the toolbar item can be toggled.

        @since 3.1.5
     */
    bool CanBeToggled() const;
};

/**
    A vector of AUI toolbar items.

    This class is actually a legacy container (see @ref overview_container for
    more details), but it can, and should be, handled as just a vector of
    wxAuiToolBarItem objects in the application code.
*/
class wxAuiToolBarItemArray : public std::vector<wxAuiToolBarItem>
{
};

/**
    @class wxAuiToolBarArt

    wxAuiToolBarArt is part of the wxAUI class framework and is the base class
    defining the interface for rendering wxAuiToolBar.

    It is an abstract base class and its concrete wxAuiGenericToolBarArt and
    wxAuiMSWToolBarArt subclasses provide the implementations used by default
    under all non-MSW platforms and under MSW respectively.

    See also @ref wxAuiToolBar and @ref overview_aui.

    @library{wxaui}
    @category{aui}
*/
class wxAuiToolBarArt
{
public:
    wxAuiToolBarArt();

    virtual wxAuiToolBarArt* Clone() = 0;
    virtual void SetFlags(unsigned int flags) = 0;
    virtual unsigned int GetFlags() = 0;
    virtual void SetFont(const wxFont& font) = 0;
    virtual wxFont GetFont() = 0;
    virtual void SetTextOrientation(int orientation) = 0;
    virtual int GetTextOrientation() = 0;

    /**
        Set the text direction for rendering text.

        Does nothing in the base class, override this function to support
        vertical text rendering.

        @since 3.3.2
     */
    virtual void SetTextDirection(wxAuiTextDirection direction);

    /**
        Return the direction used for rendering text.

        Always returns wxAuiTextDirection::LeftToRight in the base class.

        @since 3.3.2
     */
    virtual wxAuiTextDirection GetTextDirection() const;

    virtual void DrawBackground(
                         wxDC& dc,
                         wxWindow* wnd,
                         const wxRect& rect) = 0;

    virtual void DrawPlainBackground(
                                  wxDC& dc,
                                  wxWindow* wnd,
                                  const wxRect& rect) = 0;

    virtual void DrawLabel(
                         wxDC& dc,
                         wxWindow* wnd,
                         const wxAuiToolBarItem& item,
                         const wxRect& rect) = 0;

    virtual void DrawButton(
                         wxDC& dc,
                         wxWindow* wnd,
                         const wxAuiToolBarItem& item,
                         const wxRect& rect) = 0;

    virtual void DrawDropDownButton(
                         wxDC& dc,
                         wxWindow* wnd,
                         const wxAuiToolBarItem& item,
                         const wxRect& rect) = 0;

    virtual void DrawControlLabel(
                         wxDC& dc,
                         wxWindow* wnd,
                         const wxAuiToolBarItem& item,
                         const wxRect& rect) = 0;

    virtual void DrawSeparator(
                         wxDC& dc,
                         wxWindow* wnd,
                         const wxRect& rect) = 0;

    virtual void DrawGripper(
                         wxDC& dc,
                         wxWindow* wnd,
                         const wxRect& rect) = 0;

    virtual void DrawOverflowButton(
                         wxDC& dc,
                         wxWindow* wnd,
                         const wxRect& rect,
                         int state) = 0;

    /**
        Return the size of the label for the given item.

        Note that the type of @a dc was wxDC until wxWidgets 3.3.0, where it
        was changed to wxReadOnlyDC as this function doesn't modify the DC
        contents.
     */
    virtual wxSize GetLabelSize(
                         wxReadOnlyDC& dc,
                         wxWindow* wnd,
                         const wxAuiToolBarItem& item) = 0;

    /**
        Return the size of the given item.

        Note that the type of @a dc was wxDC until wxWidgets 3.3.0, where it
        was changed to wxReadOnlyDC as this function doesn't modify the DC
        contents.
     */
    virtual wxSize GetToolSize(
                         wxReadOnlyDC& dc,
                         wxWindow* wnd,
                         const wxAuiToolBarItem& item) = 0;

    /**
        Get the element size scaled by the DPI of the given window.

        This function should be used to get the size of the element in pixels.

        The default version delegates to GetElementSize(), override this
        function if a different behaviour (e.g. to use some smarter algorithm
        for scaling instead of just multiplying by the DPI factor) is needed.

        @param elementId
            One of ::wxAuiToolBarArtSetting elements.
        @param window
            A valid window, typically wxAuiToolBar itself.
        @return
            The size of the element in pixels.

        @see SetElementSize()

        @since 3.3.0
     */
    virtual int GetElementSizeForWindow(int elementId, const wxWindow* window);

    /**
        Returns the size of the given element in DIPs.

        This function is typically more convenient to override, as it can just
        return the same value as was passed to SetElementSize(), but it
        shouldn't usually be called, use GetElementSizeForWindow() instead.

        @param elementId
            One of ::wxAuiToolBarArtSetting elements.
        @return
            The size of the element in DIPs.
     */
    virtual int GetElementSize(int elementId) = 0;

    /**
        Sets the size of the given element in DIPs.

        Note that this function takes the size in DPI-independent pixels and
        this size will be scaled by the factor depending on the DPI being
        actually used by GetElementSizeForWindow(). In particular, do _not_ use
        wxWindow::FromDIP() for the @a size argument passed to this function.

        @param elementId
            One of ::wxAuiToolBarArtSetting elements.
        @param size
            The size of the element in DIPs.
     */
    virtual void SetElementSize(int elementId, int size) = 0;

    virtual int ShowDropDown(
                         wxWindow* wnd,
                         const wxAuiToolBarItemArray& items) = 0;
};

/**
    Default generic implementation of wxAuiToolBarArt.

    This class is used by default under all non-MSW platforms for rendering
    wxAuiToolBar.

    @library{wxaui}
    @category{aui}
*/
class wxAuiGenericToolBarArt : public wxAuiToolBarArt
{
public:
    /**
        Default constructor.
    */
    wxAuiGenericToolBarArt();

    /**
        Set the text direction for rendering text.

        Currently vertical text directions are only supported for non-drop down
        items.

        @since 3.3.2
     */
    virtual void SetTextDirection(wxAuiTextDirection direction) override;

    /**
        Return the direction used for rendering text.

        This is the direction last set using SetTextDirection() or
        wxAuiTextDirection::LeftToRight if it was never called.

        @since 3.3.2
     */
    virtual wxAuiTextDirection GetTextDirection() const override;
};

/**
    wxMSW-specific implementation of wxAuiToolBarArt.

    This class is available only in wxMSW port as it uses native functions and
    is used for rendering wxAuiToolBar by default in it.

    Note that native toolbar rendering functions don't respect dark mode, which
    is why this class behaves in the same way as wxAuiGenericToolBarArt when
    dark mode is used.

    @library{wxaui}
    @category{aui}
*/
class wxAuiMSWToolBarArt : public wxAuiGenericToolBarArt
{
public:
    /**
        Default constructor.
    */
    wxAuiMSWToolBarArt();

    /**
        Set the text direction for rendering text.

        Currently vertical text directions are only supported for non-drop down
        items.

        @since 3.3.2
     */
    virtual void SetTextDirection(wxAuiTextDirection direction) override;

    /**
        Return the direction used for rendering text.

        This is the direction last set using SetTextDirection() or
        wxAuiTextDirection::LeftToRight if it was never called.

        @since 3.3.2
     */
    virtual wxAuiTextDirection GetTextDirection() const override;
};


/**
    @class wxAuiDefaultToolBarArt

    wxAuiDefaultToolBarArt is part of the wxAUI class framework.
    See also @ref wxAuiToolBarArt , @ref wxAuiToolBar and @ref overview_aui.

    @library{wxaui}
    @category{aui}
*/
class wxAuiDefaultToolBarArt : public wxAuiToolBarArt
{
public:
    wxAuiDefaultToolBarArt();
    virtual ~wxAuiDefaultToolBarArt();

    virtual wxAuiToolBarArt* Clone();
    virtual void SetFlags(unsigned int flags);
    virtual unsigned int GetFlags();
    virtual void SetFont(const wxFont& font);
    virtual wxFont GetFont();
    virtual void SetTextOrientation(int orientation);
    virtual int GetTextOrientation();

    virtual void DrawBackground(
                wxDC& dc,
                wxWindow* wnd,
                const wxRect& rect);

    virtual void DrawPlainBackground(wxDC& dc,
                                  wxWindow* wnd,
                                  const wxRect& rect);

    virtual void DrawLabel(
                wxDC& dc,
                wxWindow* wnd,
                const wxAuiToolBarItem& item,
                const wxRect& rect);

    virtual void DrawButton(
                wxDC& dc,
                wxWindow* wnd,
                const wxAuiToolBarItem& item,
                const wxRect& rect);

    virtual void DrawDropDownButton(
                wxDC& dc,
                wxWindow* wnd,
                const wxAuiToolBarItem& item,
                const wxRect& rect);

    virtual void DrawControlLabel(
                wxDC& dc,
                wxWindow* wnd,
                const wxAuiToolBarItem& item,
                const wxRect& rect);

    virtual void DrawSeparator(
                wxDC& dc,
                wxWindow* wnd,
                const wxRect& rect);

    virtual void DrawGripper(
                wxDC& dc,
                wxWindow* wnd,
                const wxRect& rect);

    virtual void DrawOverflowButton(
                wxDC& dc,
                wxWindow* wnd,
                const wxRect& rect,
                int state);

    /**
        Return the size of the label for the given item.

        Note that the type of @a dc was wxDC until wxWidgets 3.3.0, where it
        was changed to wxReadOnlyDC as this function doesn't modify the DC
        contents.
     */
    virtual wxSize GetLabelSize(
                wxReadOnlyDC& dc,
                wxWindow* wnd,
                const wxAuiToolBarItem& item);

    /**
        Return the size of the given item.

        Note that the type of @a dc was wxDC until wxWidgets 3.3.0, where it
        was changed to wxReadOnlyDC as this function doesn't modify the DC
        contents.
     */
    virtual wxSize GetToolSize(
                wxReadOnlyDC& dc,
                wxWindow* wnd,
                const wxAuiToolBarItem& item);

    /**
        Return the size of the element.

        Implement the base class pure virtual function by returning the default
        element size or the last value passed to SetElementSize().
     */
    virtual int GetElementSize(int elementId);

    /**
        Change the size of the element.

        Implements the base class pure virtual function by storing the value to
        be returned by GetElementSize() and used by GetElementSizeForWindow().

        As for the base class function, @a size is in DIPs, _not_ pixels, so
        wxWindow::FromDIP() should _not_ be used for it.
     */
    virtual void SetElementSize(int elementId, int size);

    virtual int ShowDropDown(wxWindow* wnd,
                             const wxAuiToolBarItemArray& items);
};



/**
    @class wxAuiToolBar

    wxAuiToolBar is a dockable toolbar, part of the wxAUI class framework.
    See also @ref overview_aui.

    The appearance of this class is configurable and can be changed by calling
    wxAuiToolBar::SetArtProvider(). By default, native art provider is used if
    available (currently only in wxMSW) and wxAuiGenericToolBarArt otherwise.

    @beginStyleTable
    @style{wxAUI_TB_TEXT}
        Display the label strings on the toolbar buttons.
    @style{wxAUI_TB_NO_TOOLTIPS}
        Do not show tooltips for the toolbar items.
    @style{wxAUI_TB_NO_AUTORESIZE}
        Do not automatically resize the toolbar when new tools are added.
    @style{wxAUI_TB_GRIPPER}
        Show the toolbar's gripper control. If the toolbar is added to
        an AUI pane that contains a gripper, this style will be automatically
        set.
    @style{wxAUI_TB_OVERFLOW}
        Show an overflow menu containing toolbar items that can't fit on the
        toolbar if it is too small.
    @style{wxAUI_TB_VERTICAL}
        Using this style forces the toolbar to be vertical and
        be only dockable to the left or right sides of the window
        whereas by default it can be horizontal or vertical and
        be docked anywhere.
    @style{wxAUI_TB_HORZ_LAYOUT}
        For horizontal toolbars, show the text and the icons alongside,
        not vertically stacked. This style must be used with wxAUI_TB_TEXT
        and can't be used with wxAUI_TB_VERTICAL.
    @style{wxAUI_TB_HORIZONTAL}
        Analogous to wxAUI_TB_VERTICAL, but forces the toolbar
        to be horizontal.
    @style{wxAUI_TB_PLAIN_BACKGROUND}
        Draw a plain background (based on parent) instead of the
        default gradient background.
    @style{wxAUI_TB_HORZ_TEXT}
        Equivalent to wxAUI_TB_HORZ_LAYOUT | wxAUI_TB_TEXT
    @style{wxAUI_TB_DEFAULT_STYLE}
        The default is to have no styles.
    @endStyleTable

    @beginEventEmissionTable{wxAuiToolBarEvent}
    @event{EVT_AUITOOLBAR_TOOL_DROPDOWN(id, func)}
        Process a wxEVT_AUITOOLBAR_TOOL_DROPDOWN event
    @event{EVT_AUITOOLBAR_OVERFLOW_CLICK(id, func)}
        Process a wxEVT_AUITOOLBAR_OVERFLOW_CLICK event
    @event{EVT_AUITOOLBAR_RIGHT_CLICK(id, func)}
        Process a wxEVT_AUITOOLBAR_RIGHT_CLICK event
    @event{EVT_AUITOOLBAR_MIDDLE_CLICK(id, func)}
        Process a wxEVT_AUITOOLBAR_MIDDLE_CLICK event
    @event{EVT_AUITOOLBAR_BEGIN_DRAG(id, func)}
        Process a wxEVT_AUITOOLBAR_BEGIN_DRAG event
    @endEventTable

    @library{wxaui}
    @category{aui}
*/
class wxAuiToolBar : public wxControl
{
public:
    /**
        Default constructor, use Create() later.

        @since 2.9.5
     */
    wxAuiToolBar();

    /**
        Constructor creating and initializing the object.
     */
    wxAuiToolBar(wxWindow* parent,
                 wxWindowID id = wxID_ANY,
                 const wxPoint& position = wxDefaultPosition,
                 const wxSize& size = wxDefaultSize,
                 long style = wxAUI_TB_DEFAULT_STYLE);

    /**
        Really create wxAuiToolBar created using default constructor.

        @since 2.9.5
     */
    bool Create(wxWindow* parent,
                wxWindowID id = wxID_ANY,
                const wxPoint& pos = wxDefaultPosition,
                const wxSize& size = wxDefaultSize,
                long style = wxAUI_TB_DEFAULT_STYLE);
    virtual ~wxAuiToolBar();

    void SetWindowStyleFlag(long style);
    long GetWindowStyleFlag() const;

    void SetArtProvider(wxAuiToolBarArt* art);
    wxAuiToolBarArt* GetArtProvider() const;

    bool SetFont(const wxFont& font);


    ///@{
    /**
        Add a new tool to the toolbar.

        This function works similarly to the corresponding wxToolBar::AddTool()
        overloads. Note that before 3.3.0 the @a client_data parameter was not used.
     */
    wxAuiToolBarItem* AddTool(int toolId,
                 const wxString& label,
                 const wxBitmapBundle& bitmap,
                 const wxString& short_help_string = wxEmptyString,
                 wxItemKind kind = wxITEM_NORMAL);

    wxAuiToolBarItem* AddTool(int toolId,
                 const wxString& label,
                 const wxBitmapBundle& bitmap,
                 const wxBitmapBundle& disabled_bitmap,
                 wxItemKind kind,
                 const wxString& short_help_string,
                 const wxString& long_help_string,
                 wxObject* client_data);

    wxAuiToolBarItem* AddTool(int toolId,
                 const wxBitmapBundle& bitmap,
                 const wxBitmapBundle& disabled_bitmap,
                 bool toggle = false,
                 wxObject* client_data = nullptr,
                 const wxString& short_help_string = wxEmptyString,
                 const wxString& long_help_string = wxEmptyString);
    ///@}

    wxAuiToolBarItem* AddLabel(int toolId,
                  const wxString& label = wxEmptyString,
                  const int width = -1);
    wxAuiToolBarItem* AddControl(wxControl* control,
                    const wxString& label = wxEmptyString);
    wxAuiToolBarItem* AddSeparator();
    wxAuiToolBarItem* AddSpacer(int pixels);
    wxAuiToolBarItem* AddStretchSpacer(int proportion = 1);

    bool Realize();

    wxControl* FindControl(int window_id);
    wxAuiToolBarItem* FindToolByPosition(wxCoord x, wxCoord y) const;
    wxAuiToolBarItem* FindToolByIndex(int idx) const;
    wxAuiToolBarItem* FindTool(int toolId) const;

    void ClearTools();
    void Clear();

    /**
        Destroys the tool with the given ID and its associated window, if any.

        @param toolId ID of a previously added tool.
        @return @true if the tool was destroyed or @false otherwise, e.g. if
            the tool with the given ID was not found.

        @since 3.1.4
     */
    bool DestroyTool(int toolId);

    /**
        Destroys the tool at the given position and its associated window, if
        any.

        @param idx The index, or position, of a previously added tool.
        @return @true if the tool was destroyed or @false otherwise, e.g. if
            the provided index is out of range.
     */
    bool DestroyToolByIndex(int idx);

    /**
        Removes the tool with the given ID from the toolbar.

        Note that if this tool was added by AddControl(), the associated
        control is @e not deleted and must either be reused (e.g. by
        reparenting it under a different window) or destroyed by caller.
        If this behaviour is unwanted, prefer using DestroyTool() instead.

        @param toolId ID of a previously added tool.
        @return @true if the tool was removed or @false otherwise, e.g. if the
            tool with the given ID was not found.
     */
    bool DeleteTool(int toolId);

    /**
        Removes the tool at the given position from the toolbar.

        Note that if this tool was added by AddControl(), the associated
        control is @e not deleted and must either be reused (e.g. by
        reparenting it under a different window) or destroyed by caller.
        If this behaviour is unwanted, prefer using DestroyToolByIndex()
        instead.

        @param idx The index, or position, of a previously added tool.
        @return @true if the tool was removed or @false otherwise, e.g. if the
            provided index is out of range.
     */
    bool DeleteByIndex(int idx);

    size_t GetToolCount() const;
    int GetToolPos(int toolId) const;
    int GetToolIndex(int toolId) const;
    bool GetToolFits(int toolId) const;
    wxRect GetToolRect(int toolId) const;
    bool GetToolFitsByIndex(int toolId) const;
    bool GetToolBarFits() const;

    void SetMargins(const wxSize& size);
    void SetMargins(int x, int y);
    void SetMargins(int left, int right, int top, int bottom);

    /**
        Associates a wxObject with the item identified by id.

        @param toolId Identifier of the desired item
        @param clientData Pointer to the wxObject

        @see GetToolClientData()

        @since 3.3.0
    */
    void SetToolClientData(int toolId, wxObject* clientData);

    /**
        Get wxObject associated with the item identified by id.

        @param toolId Identifier of the desired item
        @return Pointer to the associated wxObject

        @see SetToolClientData()

        @since 3.3.0
    */
    wxObject* GetToolClientData(int toolId) const;

    void SetToolBitmapSize(const wxSize& size);
    wxSize GetToolBitmapSize() const;

    bool GetOverflowVisible() const;
    void SetOverflowVisible(bool visible);

    bool GetGripperVisible() const;
    void SetGripperVisible(bool visible);

    void ToggleTool(int toolId, bool state);
    bool GetToolToggled(int toolId) const;

    void EnableTool(int toolId, bool state);
    bool GetToolEnabled(int toolId) const;

    /**
        Set whether the specified toolbar item has a drop down button.

        This is only valid for wxITEM_NORMAL tools.

        @see wxAuiToolBarItem::SetHasDropDown()
    */
    void SetToolDropDown(int toolId, bool dropdown);

    /**
        Returns whether the specified toolbar item has an associated drop down
        button.

        @see wxAuiToolBarItem::HasDropDown()
    */
    bool GetToolDropDown(int toolId) const;

    void SetToolBorderPadding(int padding);
    int  GetToolBorderPadding() const;

    void SetToolTextOrientation(int orientation);
    int  GetToolTextOrientation() const;

    void SetToolPacking(int packing);
    int  GetToolPacking() const;

    void SetToolProportion(int toolId, int proportion);
    int  GetToolProportion(int toolId) const;

    /**
        Set the tool separation in DIPs.

        Please note that because this function passes @a separation to
        wxAuiToolBarArt::SetElementSize() it should be given in DIPs, not in
        (logical) pixels. I.e. do _not_ use wxWindow::FromDIP() for this
        function argument.
     */
    void SetToolSeparation(int separation);

    /**
        Returns the separation between tools in logical pixels.
     */
    int GetToolSeparation() const;

    void SetToolSticky(int toolId, bool sticky);
    bool GetToolSticky(int toolId) const;

    wxString GetToolLabel(int toolId) const;
    void SetToolLabel(int toolId, const wxString& label);

    wxBitmap GetToolBitmap(int toolId) const;
    void SetToolBitmap(int toolId, const wxBitmapBundle& bitmap);

    wxString GetToolShortHelp(int toolId) const;
    void SetToolShortHelp(int toolId, const wxString& help_string);

    wxString GetToolLongHelp(int toolId) const;
    void SetToolLongHelp(int toolId, const wxString& help_string);

    /**
      Add toolbar items that are always displayed in the overflow menu.

      If there are custom items set, then the overflow menu will be
      displayed even if there are no items from the main toolbar that
      overflow.

      @param prepend are the items to show before any overflow items
      @param append are the items to show after any overflow items

      @note The toolbar must have the @c wxAUI_TB_OVERFLOW style.
     */
    void SetCustomOverflowItems(const wxAuiToolBarItemArray& prepend,
                                const wxAuiToolBarItemArray& append);

    /** get size of hint rectangle for a particular dock location */
    wxSize GetHintSize(int dock_direction) const;
    bool IsPaneValid(const wxAuiPaneInfo& pane) const;
};

